#include "burnint.h"
#include "burn_sound.h"

#define DAC2_NUM		(8)	// Maximum DAC chips

struct dac2_info
{
	INT16	Output;
	double 	nVolume;
	INT32 	nCurrentPosition;
	INT32	Initialized;
	INT32	OutputDir;
	INT32	(*pSyncCallback)();
};

static struct dac2_info dac2_table[DAC2_NUM];

static INT16 UnsignedVolTable[256];
static INT16 SignedVolTable[256];

static INT16 *lBuffer = NULL;
static INT16 *rBuffer = NULL;

static INT32 NumChips;

static INT32 bAddSignal;

static void UpdateStream(INT32 chip, INT32 length)
{
	struct dac2_info *ptr;

	if (lBuffer == NULL) {	// delay buffer allocation for cases when fps is not 60
		lBuffer = (INT16*)BurnMalloc(nBurnSoundLen * sizeof(INT16));
		memset (lBuffer, 0, nBurnSoundLen * sizeof(INT16));
	}
	if (rBuffer == NULL) {	// delay buffer allocation for cases when fps is not 60
		rBuffer = (INT16*)BurnMalloc(nBurnSoundLen * sizeof(INT16));
		memset (rBuffer, 0, nBurnSoundLen * sizeof(INT16));
	}

        ptr = &dac2_table[chip];
        if (ptr->Initialized == 0) return;

        if (length > nBurnSoundLen) length = nBurnSoundLen;
        length -= ptr->nCurrentPosition;
        if (length <= 0) return;

        INT16 *lbuf = lBuffer + ptr->nCurrentPosition;
	INT16 *rbuf = rBuffer + ptr->nCurrentPosition;

	INT16 lOut = ((ptr->OutputDir & BURN_SND_ROUTE_LEFT ) == BURN_SND_ROUTE_LEFT ) ? ptr->Output : 0;
        INT16 rOut = ((ptr->OutputDir & BURN_SND_ROUTE_RIGHT) == BURN_SND_ROUTE_RIGHT) ? ptr->Output : 0;

        ptr->nCurrentPosition += length;

        if (rOut && lOut) {              
                while (length--) {
                        *lbuf++ = *lbuf + lOut;
			*rbuf++ = *rbuf + rOut;
                }
        } else if (lOut) {              
                while (length--) *lbuf++ = *lbuf + lOut;
        } else if (rOut) {            
                while (length--) *rbuf++ = *rbuf + rOut;
        }
}

void DAC2Update(INT16* Buffer, INT32 Length)
{
	struct dac2_info *ptr;

	for (INT32 i = 0; i < NumChips; i++) {
		UpdateStream(i, nBurnSoundLen);
	}

	INT16 *lbuf = lBuffer;
	INT16 *rbuf = rBuffer;

	if (bAddSignal) {
		while (Length--) {
			Buffer[0] = BURN_SND_CLIP((INT32)(lbuf[0] + Buffer[0]));
			Buffer[1] = BURN_SND_CLIP((INT32)(rbuf[0] + Buffer[1]));
			Buffer += 2;
			lbuf[0] = 0; // clear buffer
			rbuf[0] = 0; // clear buffer
			lbuf++;
			rbuf++;
		}
	} else {
		while (Length--) {
			Buffer[0] = lbuf[0];
			Buffer[1] = rbuf[0];
			Buffer += 2;
			lbuf[0] = 0; // clear buffer
			rbuf[0] = 0; // clear buffer
			lbuf++;
			rbuf++;
		}
	}

	for (INT32 i = 0; i < NumChips; i++) {
		ptr = &dac2_table[i];
		ptr->nCurrentPosition = 0;
	}
}

void DAC2Write(INT32 Chip, UINT8 Data)
{
	struct dac2_info *ptr;

	ptr = &dac2_table[Chip];

	UpdateStream(Chip, ptr->pSyncCallback());

	ptr->Output = (INT32)(UnsignedVolTable[Data] * ptr->nVolume);
}

void DAC2Write16(INT32 Chip, INT16 Data)
{
	
	struct dac2_info *ptr;

	ptr = &dac2_table[Chip];

	if (Data != ptr->Output) {
		UpdateStream(Chip, ptr->pSyncCallback());
		ptr->Output = Data;
	}
}


void DAC2SignedWrite(INT32 Chip, UINT8 Data)
{
	struct dac2_info *ptr;

	ptr = &dac2_table[Chip];

	UpdateStream(Chip, ptr->pSyncCallback());

	ptr->Output = (INT32)(SignedVolTable[Data] * ptr->nVolume);
}

static void DAC2BuildVolTables()
{
	for (INT32 i = 0;i < 256;i++) {
		UnsignedVolTable[i] = i * 0x101 / 2;
		SignedVolTable[i] = i * 0x101 - 0x8000;
	}
}

void DAC2Init(INT32 Num, UINT32 /*Clock*/, INT32 bAdd, INT32 (*pSyncCB)())
{
	struct dac2_info *ptr;

	//DebugSnd_DACInitted = 1;
	
	NumChips = Num + 1;

	ptr = &dac2_table[Num];

	memset (ptr, 0, sizeof(dac2_info));

	ptr->Initialized = 1;
	ptr->nVolume = 1.00;
	ptr->OutputDir = BURN_SND_ROUTE_BOTH;
	ptr->pSyncCallback = pSyncCB;

	DAC2BuildVolTables(); // necessary to build for every chip?
	
	bAddSignal = bAdd;
}

void DAC2SetRoute(INT32 Chip, double nVolume, INT32 nRouteDir)
{
	struct dac2_info *ptr;

	ptr = &dac2_table[Chip];
	ptr->nVolume = nVolume;
	ptr->OutputDir = nRouteDir;
}

void DAC2Reset()
{
	struct dac2_info *ptr;

	for (INT32 i = 0; i < NumChips; i++) {
		ptr = &dac2_table[i];

		ptr->nCurrentPosition = 0;
		ptr->Output = 0;
	}
}

void DAC2Exit()
{
	struct dac2_info *ptr;

	for (INT32 i = 0; i < DAC2_NUM; i++) {
		ptr = &dac2_table[i];

		ptr->Initialized = 0;
		ptr->pSyncCallback = NULL;
	}

	NumChips = 0;
	
	//DebugSnd_DACInitted = 0;

	BurnFree (lBuffer);
	BurnFree (rBuffer);
	lBuffer = NULL;
	rBuffer = NULL;
}

INT32 DAC2Scan(INT32 nAction,INT32 *pnMin)
{
	if (pnMin != NULL) {
		*pnMin = 0x029719;
	}
	
	struct dac2_info *ptr;

	if (nAction & ACB_DRIVER_DATA) {
		for (INT32 i = 0; i < NumChips; i++) {
			ptr = &dac2_table[i];

			SCAN_VAR(ptr->Output);
		}
	}

	return 0;
}
